/***************************************************************************

The module converts an RGB value to a CRY value.

The routine is as follows:

void RGB2CRY(byte Rb, byte Gb, byte Bb, byte *Intensity, byte *Color)

Input:   Rb, Gb, Bb		Levels for Red, Green, and Blue each from 0-255
Output:  Intensity		CRY Intensity value 0-255
Output:  Color				CRY Color value 0-255

Pointers to Intesity and Color are passed to RGB2CRY so they can be set.

Since Rb, Gb, Bb are bytes, and bytes are always legal input to RGB2CRY,
there is no error condition.

See main() for an example.

---------------------------------------------------------------------------

Note that the memory-map for CRY is two bytes per pixel in this order:

Intensity,Color

---------------------------------------------------------------------------

This module uses Find_Best_Match() to find the best CRY Color value for any
given R,G,B ratio ("R,G,B ratio" being any R,G,B where at least one value
is exactly 255, and the other two are in the range 0-255). Find_Best_Match
compairs the R,G,B values of each CRY Color to the R,G,B input, and chooses
the best fit. This is probably the most accurate method for converting a 
pixel, but it's certainly not the fastest. 

The CRY Intensity for any R,G,B is the greatest of the three values.

Andrew: We were working on a fast, table-driven method. It would be a lot
faster- we don't know how much accuracy we'd lose because we never finished
it. If you want that code instead, let us know.

---------------------------------------------------------------------------

Copyright (C) 1993 Atari Corp.

Written by Mike Pooler (x8805)

***************************************************************************/

#include <stdio.h>

typedef unsigned char byte;
typedef unsigned short word;

# include "crytable.c" /* The RGB combos */

byte Find_Best_Match(R,G,B)
		byte	R;
		byte	G;
		byte	B;
{
	/* Find one offset so that Red[Ofs] Green[Ofs] Blue[Ofs] */
	/* matches R G B as closely as possible */
	byte *Table1;
	byte *Table2;
	byte *Table_255;
	byte Val1, Val2;
	long Diff;
	word Best_Diff = 0xFFFF;
	byte Best_Ofs;
	short x;
	
	/* One value will always be 255, do the searching on the other tables */
	if      (R == 255) { Table1 = Green; Val1 = G;	Table2 = Blue; Val2 = B; Table_255 = Red;   }
	else if (G == 255) { Table1 = Red;   Val1 = R;	Table2 = Blue; Val2 = B; Table_255 = Green; }
	else if (B == 255) { Table1 = Green; Val1 = G;	Table2 = Red;  Val2 = R; Table_255 = Blue;  }

	for (x = 0; x < 256; x++)
	{
		if (Table_255[x] == 255)
		{
			Diff  = abs((short)Table1[x] - (short)Val1);

			if (Diff <= Best_Diff)
			{
				Diff += abs((short)Table2[x] - (short)Val2);
		
				if (Diff < Best_Diff) 
				{
					if (Diff == 0)
					{
						return((byte)x);
					}
					Best_Diff = Diff; 
					Best_Ofs = x; 
				}
			}
		}
	}
		
	return(Best_Ofs);
}

void RGB2CRY(Rb,Gb,Bb,Intensity,Color)
		byte	Rb;
		byte	Gb;
		byte	Bb;
		byte	*Intensity;
		byte	*Color;
{
	long R, G, B, I, Mult;
	byte Offset;

	Mult = 0;
	
	I = Rb;
	if (Gb > I) I = Gb;
	if (Rb > I) I = Rb;

	*Intensity = (byte)I;

	/* Zero intensity is the same as no color	*/
	if (*Intensity == 0) 
	{
		*Color = 0;
	}
	else
	{
		R = (long)Rb;
		G = (long)Gb;
		B = (long)Bb;
	
		fprintf( stdout, "Step #1: R %ld G %ld B %ld\n", R, G, B );
		fflush( stdout );

		Mult = ((long)(0xFF << 16)) / I;

		fprintf( stdout, "I: %ld, Mult: %ld\n", I, Mult );
		fflush( stdout );

		R *= Mult;
		G *= Mult;
		B *= Mult;

		fprintf( stdout, "Step #2: R %ld G %ld B %ld\n", R, G, B );
		fflush( stdout );

		R += 0x8000; /* Round up if reminader > 0x7FFF */
		G += 0x8000;
		B += 0x8000;
		R >>= 16; G >>= 16; B >>= 16;
		
		fprintf( stdout, "Step #3: R %ld G %ld B %ld\n", R, G, B );
		fflush( stdout );

		fprintf( stdout, "R %d G %d B %d\n", (byte)R, (byte)G, (byte)B );
		fflush( stdout );

		Offset = Find_Best_Match((byte)R, (byte)G, (byte)B); 
	
		fprintf( stdout, "Offset %d\n", Offset );
		fflush( stdout );

		*Color = Offset;	
	}
}

void RGB2CRY_float(R,G,B,Intensity,Color)
		float	R;
		float	G;
		float	B;
		byte	*Intensity;
		byte	*Color;
	{
	float I;
	
	I = B;
	if (G > I) I = G;
	if (R > I) I = R;
	
	*Intensity = (byte)((I * 255.0) + 0.49);

	if (*Intensity == 0) 
		{
		*Color = 0;
		return;
		}

	R = (byte)(((R / I) * 255.0) + 0.49);
	G = (byte)(((G / I) * 255.0) + 0.49);
	B = (byte)(((B / I) * 255.0) + 0.49);
					
	*Color = Find_Best_Match((byte)R, (byte)G, (byte)B); 
	}

#if 0
short main()
	{
	byte Color, Intensity;
	byte R = 80;
	byte G = 120;
	byte B = 160;
	float Rf = 80.0  / 255.0;
	float Gf = 120.0 / 255.0;
	float Bf = 160.0 / 255.0;
	
	RGB2CRY(R, G, B, &Intensity, &Color);
	printf("\nFor R=%d, G=%d, B=%d: Intesity = %d and Color = %d\n\n", 
	          R, G, B, Intensity, Color);
				 
	RGB2CRY_float(Rf, Gf, Bf, &Intensity, &Color);
	printf("\nFor R=%f, G=%f, B=%f: Intesity = %d and Color = %d\n\n",
	          Rf, Gf, Bf, Intensity, Color);
	return(0);
	}
#endif
